﻿using GeneticMultistepSG.Struct;
using System;
using System.Collections.Generic;
using System.Linq;

namespace GeneticMultistepSG
{
    /// <summary> Strategie w chromosomie reprezentowane są jako lista strategii prostych obrońcy + ich prawdopodobieństwa
    /// </summary>
	public class ChromosomeList : Chromosome
	{
        /// <summary> Lista strategii mieszanych dla poszczególnych obrońców
        /// </summary>
        public DefenderStrategy[] defenderStrategies;


		/// <summary>
		/// Towrzy głęboką kopię chromosomu
		/// </summary>
		public override Chromosome MakeCopy()
		{
			ChromosomeList result = new ChromosomeList();
			result.fittingFunction = fittingFunction;
			result.fittingFunctionSecondStage = fittingFunctionSecondStage;
			result.attackerResult = attackerResult;
			if (attackStrategy == null)
				result.attackStrategy = null;
			else
				result.attackStrategy = attackStrategy.Select(x => x).ToList();
            result.boundedRationality = boundedRationality.Select(x => x).ToList();

			result.defenderStrategies = new DefenderStrategy[defenderStrategies.Length];
			for (int i = 0; i < defenderStrategies.Length; i++)
				result.defenderStrategies[i] = defenderStrategies[i].Copy();

            result.isDifferentAttacker = isDifferentAttacker;

            return result;
		}

        private void Restore(ChromosomeList c)
        {
            fittingFunction = c.fittingFunction;
            fittingFunctionSecondStage = c.fittingFunctionSecondStage;
            attackerResult = c.attackerResult;
            if (c.attackStrategy == null)
                attackStrategy = null;
            else
                attackStrategy = c.attackStrategy.Select(x => x).ToList();
            boundedRationality = c.boundedRationality.Select(x => x).ToList();

            defenderStrategies = new DefenderStrategy[c.defenderStrategies.Length];
            for (int i = 0; i < c.defenderStrategies.Length; i++)
                defenderStrategies[i] = c.defenderStrategies[i].Copy();

            isDifferentAttacker = c.isDifferentAttacker;
        }


        /// <summary>
        /// Inicjalizuje chromosom (tworzy losową strategię)
        /// </summary>
        public override void Init()
        {
            defenderStrategies = new DefenderStrategy[Program.gameDefinition.defenderUnitCount];
            for (int j = 0; j < defenderStrategies.Length; j++)
            {
                defenderStrategies[j] = new DefenderStrategy();
                defenderStrategies[j].elements.Add(new int[Program.gameDefinition.rounds + 1]);

                defenderStrategies[j].elements[0][0] = 0;

                //losowe strategie początkowe
                for (int k = 1; k < defenderStrategies[j].elements[0].Length; k++)
                {
                    int prevPosition = defenderStrategies[j].elements[0][k - 1];
                    defenderStrategies[j].elements[0][k] = MoveDefenderRandomly(defenderStrategies[j].elements[0][k - 1]);
                    //TODO: rozważyć preferowanie strategi inicjalnych z pokryciem celów bardziej niż zupełnie losowe
                }

                defenderStrategies[j].probabilities.Add(1.0);
                defenderStrategies[j].partialPayoffs.Add(0);
            }
            
          
        }



        /// <summary>
        /// Wyliczenie wypłaty atakującego (attackerResult) i obrońcy (defenderResult) dla strategii atakującego attackerStrategy,
        /// z uwględnieniem ograniczonej racjonalności brStrategy. Strategia obrońcy z chromosomu.
        /// </summary>
		public override void CalculatePayoff(List<int> attackerStrategy, List<BoundedRationalityStrategy> brStrategy, out double attackerResult, out double defenderResult)
		{
			attackerResult = 0.0; defenderResult = 0.0;
			double currentProbablility = 1.0;

            DefenderStrategy[] defenderStrategiesToEvaluate = new DefenderStrategy[defenderStrategies.Length];

            if (brStrategy.Contains(BoundedRationalityStrategy.AnchoringTheoryProbabilityApproximation))
            {
                for (int i = 0; i < defenderStrategies.Length; i++)
                    defenderStrategiesToEvaluate[i] = BoundedRationality.AnchoringTheoryProbabilityApproximation(defenderStrategies[i]);
            }
            else if (brStrategy.Contains(BoundedRationalityStrategy.AnchoringTheoryProbability))
            {
                for (int i = 0; i < defenderStrategies.Length; i++)
                    defenderStrategiesToEvaluate[i] = BoundedRationality.AnchoringTheoryProbability(defenderStrategies[i]);
            }
            else
                defenderStrategiesToEvaluate = defenderStrategies;

			for (int interval = 0; interval < attackerStrategy.Count; interval++)
			{
				int v = attackerStrategy[interval];

				double probabilityAttackerCaught = 0.0;
                for (int j = 0; j < defenderStrategiesToEvaluate[0].elements.Count; j++) //po strategiach prostych obrońców (dla każdego obrońcy jest tyle samo strategii prostych w strategii mieszanej i są z tym samym prawdopodobieństwem)
				{
					bool isCaught = false;
                    for (int i = 0; i < defenderStrategiesToEvaluate.Length; i++) //po obrońcach
					{
                        if (defenderStrategiesToEvaluate[i].elements[j][interval] == v)
							isCaught = true;
					}
					if (isCaught)
                        probabilityAttackerCaught += defenderStrategiesToEvaluate[0].probabilities[j];
				}

				double defenderReward = Program.gameDefinition.vertexDefenderRewards[v];
				double attackerPenality = Program.gameDefinition.vertexAttackerPenalties[v];

				if (brStrategy.Contains(BoundedRationalityStrategy.ProspectTheoryPayoff))
				{
					defenderReward = BoundedRationality.ProspectTheoryPayoff(defenderReward);
					attackerPenality = BoundedRationality.ProspectTheoryPayoff(attackerPenality);
				}

				defenderResult += currentProbablility * probabilityAttackerCaught * defenderReward;
				attackerResult += currentProbablility * probabilityAttackerCaught * attackerPenality;

				if (Program.gameDefinition.targets.Contains(v)) //znajdujemy się w celu
				{
					double defenderPanality = Program.gameDefinition.targetDefenderPenalties[Program.gameDefinition.targets.IndexOf(v)];
					double attackerReward = Program.gameDefinition.targetAttackerRewards[Program.gameDefinition.targets.IndexOf(v)];

					if (brStrategy.Contains(BoundedRationalityStrategy.ProspectTheoryPayoff))
					{
						defenderPanality = BoundedRationality.ProspectTheoryPayoff(defenderPanality);
						attackerReward = BoundedRationality.ProspectTheoryPayoff(attackerReward);
					}

					defenderResult += currentProbablility * (1 - probabilityAttackerCaught) * defenderPanality;
					attackerResult += currentProbablility * (1 - probabilityAttackerCaught) * attackerReward;
				}

				currentProbablility = currentProbablility * (1 - probabilityAttackerCaught); //prawdopodobieństwo, że atakujący nie został złapany i idzie dalej

			}
		}


        public override void CalculatePartialPayoffs(List<int> attackerStrategy)
        {
            defenderStrategies[0].partialPayoffs = new List<double>();
            for (int i=0; i<defenderStrategies[0].probabilities.Count; i++)
            {
                defenderStrategies[0].partialPayoffs.Add(0);
                for (int interval = 0; interval < attackerStrategy.Count; interval++)
                {
                    int attackerV = attackerStrategy[interval];
                    int defenderV = defenderStrategies[0].elements[i][interval];

                    if (attackerV == defenderV)
                    {
                        defenderStrategies[0].partialPayoffs[i] = (Program.gameDefinition.vertexDefenderRewards[attackerV]);
                        break;
                    }
                    else if (Program.gameDefinition.targets.Contains(attackerV))
                    {
                        defenderStrategies[0].partialPayoffs[i] = (Program.gameDefinition.targetDefenderPenalties[Program.gameDefinition.targets.IndexOf(attackerV)]);
                        break;
                    }
                }
            }
        }


        public override void Mutate()
		{
            Program.population.mutationCount++;

            double payoffBeforeMutation = fittingFunction;
            double payoffAfterMutation = fittingFunction;

            ChromosomeList original = (ChromosomeList)this.MakeCopy();

            int i = 0;
            do
            {
                Restore(original);

                MutateStandard();

                if (Program.config.mutationAddPureStrategy)
                    MutateAddPureStrategy();

                if (Program.config.mutationDeletePureStrategy)
                    MutateDeletePureStrategy();

                if (Program.config.mutationChangeProbability)
                    MutateChangeProbability();

                if (Program.config.mutationSwitchProbability)
                    MutateSwitchProbabilities();

                Evaluate();
                payoffAfterMutation = fittingFunction;
                i++;
            }
            while (Program.config.mutationBetterPayoff && i < Program.mutationsRepeat && payoffBeforeMutation >= payoffAfterMutation);

            if (payoffAfterMutation > payoffBeforeMutation)
                Program.population.mutationIncreasePayoff++;
        }


        /// <summary>
        /// Przeprowadza mutację chromosomu.
        /// Jeden losowy obrońca zmienia losowo położenie w jednej losowej strategii od losowego interwału czasowego.
        /// </summary>
        public void MutateStandard()
        {
            DefenderStrategy defenderToMute = defenderStrategies[Program.rand.Next(defenderStrategies.Length)];
            int strategyToMuteIndex = GetPureStrategyIndxToMute();

            int firstIntervalToMute = Program.rand.Next(defenderToMute.elements[strategyToMuteIndex].Length - 1) + 1;
            for (int j = firstIntervalToMute; j < defenderToMute.elements[strategyToMuteIndex].Length; j++)
            {
                if (j > 0) //mutujemy tylko pozycję w niezerowym interwale czasowym - w zerowym startujemy z ustalonego wierzchołka
                    defenderToMute.elements[strategyToMuteIndex][j] = MoveDefenderRandomly(defenderToMute.elements[strategyToMuteIndex][j - 1]);
            }
        }

        public void MutateSwitchProbabilities()
        {
            int i1 = Program.rand.Next(defenderStrategies[0].probabilities.Count);
            int i2 = Program.rand.Next(defenderStrategies[0].probabilities.Count);

            double tmp = defenderStrategies[0].probabilities[i1];
            defenderStrategies[0].probabilities[i1] = defenderStrategies[0].probabilities[i2];
            defenderStrategies[0].probabilities[i2] = tmp;
        }

        public void MutateChangeProbability()
        {
            int indx = GetPureStrategyIndxToMute();
            defenderStrategies[0].probabilities[indx] = Program.rand.NextDouble();
            defenderStrategies[0].NormalizeProbabilities();
        }

        public void MutateAddPureStrategy()
        {
            defenderStrategies[0].elements.Add(new int[Program.gameDefinition.rounds + 1]);
            defenderStrategies[0].elements.Last()[0] = 0;

            //losowe strategie początkowe
            for (int k = 1; k < defenderStrategies[0].elements.Last().Length; k++)
            {
                int prevPosition = defenderStrategies[0].elements.Last()[k - 1];
                defenderStrategies[0].elements.Last()[k] = MoveDefenderRandomly(defenderStrategies[0].elements.Last()[k - 1]);
            }

            defenderStrategies[0].probabilities.Add(Program.rand.NextDouble());
            defenderStrategies[0].partialPayoffs.Add(0);
            defenderStrategies[0].NormalizeProbabilities();
        }

        public void MutateDeletePureStrategy()
        {
            if (defenderStrategies[0].elements.Count <= 1) return;
            int indx = GetPureStrategyIndxToMute();
            defenderStrategies[0].elements.RemoveAt(indx);
            defenderStrategies[0].probabilities.RemoveAt(indx);
            defenderStrategies[0].partialPayoffs.RemoveAt(indx);

            defenderStrategies[0].NormalizeProbabilities();
        }

        public int GetPureStrategyIndxToMute()
        {
            if (Program.config.mutationWeakestPureStrategy)
            {
                int minI = 0;
                for (int i = 0; i < defenderStrategies[0].partialPayoffs.Count; i++)
                    if (defenderStrategies[0].partialPayoffs[i] < defenderStrategies[0].partialPayoffs[minI])
                        minI = i;
                return minI;
            }

            if (Program.config.mutationWeakestPureStrategyProportional)
            {
                double maxPayoff = defenderStrategies[0].partialPayoffs.Max();
                double minPayoff = defenderStrategies[0].partialPayoffs.Min();

                if (minPayoff == maxPayoff)
                    return Program.rand.Next(defenderStrategies[0].elements.Count);

                double partialPayoffsNormalizedSum = defenderStrategies[0].partialPayoffs.Select(x=>(x-minPayoff)/(maxPayoff-minPayoff)).Sum();
                double rand = Program.rand.NextDouble() * partialPayoffsNormalizedSum;

                double partialSum = 0.0;
                for (int i = 0; i < defenderStrategies[0].partialPayoffs.Count; i++)
                {
                    partialSum += (defenderStrategies[0].partialPayoffs[i] - minPayoff) / (maxPayoff - minPayoff);
                    if (partialSum >= rand)
                        return i;
                }
            }

            return Program.rand.Next(defenderStrategies[0].elements.Count);
        }


        /// <summary>
        /// Lokalna optymalizacja
        /// </summary>
        public override void LocalOptimization()
		{
			DefenderStrategy defenderToMute = defenderStrategies[Program.rand.Next(defenderStrategies.Length)];
			int strategyToMuteIndex = Program.rand.Next(defenderToMute.elements.Count);

			for (int j = 1; j < defenderToMute.elements[strategyToMuteIndex].Length - 1; j++)
			{
				int vPrev = defenderToMute.elements[strategyToMuteIndex][j - 1];
				int vNext = defenderToMute.elements[strategyToMuteIndex][j - 1];

				List<int> targetsToConsider = Program.gameDefinition.graphConfig.adjacencyList[vPrev].Intersect(Program.gameDefinition.targets)
					.Where(x => Program.gameDefinition.graphConfig.adjacencyList[x].Contains(vNext)).ToList();
				if (targetsToConsider.Count > 0)
					defenderToMute.elements[strategyToMuteIndex][j] = targetsToConsider[Program.rand.Next(targetsToConsider.Count)];
			}
		}



        public override Chromosome Crossover(Chromosome c1, Chromosome c2, int version = 0)
        {
            Program.population.crossoverCount++;
            if (version == 0)
                return Crossover0(c1 as ChromosomeList, c2 as ChromosomeList);
            else if (version == 1)
                return Crossover1(c1 as ChromosomeList, c2 as ChromosomeList);
            else throw new NotImplementedException("Brak krzyzowania w wersji " + version);
        }

        /// <summary>
        /// Krzyżowanie dwóch chromosomów
        /// </summary>
        public ChromosomeList Crossover0(ChromosomeList c1, ChromosomeList c2)
        {
            List<double> randomNumbers = new List<double>(); //lista losowych niewielkich liczb, które są używane do usuwania strategi z małym prawdopodobieństwem, potrzebna lista na początku, aby każdy obrońca miał takie same prawdopodobieństwa
            for (int i = 0; i < c1.defenderStrategies[0].probabilities.Count + c2.defenderStrategies[0].probabilities.Count; i++)
                randomNumbers.Add(Program.rand.NextDouble() * Program.rand.NextDouble());

            ChromosomeList result = new ChromosomeList();
            result.defenderStrategies = new DefenderStrategy[c1.defenderStrategies.Length];
            for (int i = 0; i < c1.defenderStrategies.Length; i++) //dla każdego z obrońców
            {
                //łączymy strategie z dwóch chromosomów, a prawdopodobieństwa dzielimy na 2
                result.defenderStrategies[i] = new DefenderStrategy();
                DefenderStrategy c1Strategy = c1.defenderStrategies[i].Copy();
                for (int j = 0; j < c1Strategy.elements.Count; j++)
                {
                    result.defenderStrategies[i].elements.Add(c1Strategy.elements[j]);
                    result.defenderStrategies[i].probabilities.Add(c1Strategy.probabilities[j] / 2);
                    result.defenderStrategies[i].partialPayoffs.Add(c1Strategy.partialPayoffs[j]);
                }
                DefenderStrategy c2Strategy = c2.defenderStrategies[i].Copy();
                for (int j = 0; j < c2Strategy.elements.Count; j++)
                {
                    result.defenderStrategies[i].elements.Add(c2Strategy.elements[j]);
                    result.defenderStrategies[i].probabilities.Add(c2Strategy.probabilities[j] / 2);
                    result.defenderStrategies[i].partialPayoffs.Add(c2Strategy.partialPayoffs[j]);
                }

                //usuwamy strategie z strategii mieszanych z prawdopodobieństem odwrotnym do prawdopodobieństwa ich wyboru
                List<int> strategiesToRemove = new List<int>(); //lista indeksów strategii z strategii mieszanych, które usuwamy
                for (int j = 0; j < result.defenderStrategies[i].probabilities.Count; j++)
                    if (randomNumbers[j] > result.defenderStrategies[i].probabilities[j] && result.defenderStrategies[i].probabilities[j] < result.defenderStrategies[i].probabilities.Max()) //maksymalnej nie usuwamy
                        strategiesToRemove.Add(j);

                result.defenderStrategies[i] = result.defenderStrategies[i].Copy(strategiesToRemove);
            }

            return result;
        }

        /// <summary>
        /// Krzyżowanie dwóch chromosomów - wersja 2, usuwamy losowe strategie
        /// </summary>
        public ChromosomeList Crossover1(ChromosomeList c1, ChromosomeList c2)
        {
            List<double> randomNumbers = new List<double>(); //lista losowych niewielkich liczb, które są używane do usuwania strategi z małym prawdopodobieństwem, potrzebna lista na początku, aby każdy obrońca miał takie same prawdopodobieństwa
            bool isOneLessThan05 = false;

            while (!isOneLessThan05)
            {
                randomNumbers = new List<double>();
                for (int i = 0; i < c1.defenderStrategies[0].probabilities.Count + c2.defenderStrategies[0].probabilities.Count; i++)
                {
                    randomNumbers.Add(Program.rand.NextDouble());
                    if (randomNumbers.Last() < 0.5)
                        isOneLessThan05 = true;
                }
            }

            ChromosomeList result = new ChromosomeList();
            result.defenderStrategies = new DefenderStrategy[c1.defenderStrategies.Length];
            for (int i = 0; i < c1.defenderStrategies.Length; i++) //dla każdego z obrońców
            {
                //łączymy strategie z dwóch chromosomów, a prawdopodobieństwa dzielimy na 2
                result.defenderStrategies[i] = new DefenderStrategy();
                DefenderStrategy c1Strategy = c1.defenderStrategies[i].Copy();
                for (int j = 0; j < c1Strategy.elements.Count; j++)
                {
                    result.defenderStrategies[i].elements.Add(c1Strategy.elements[j]);
                    result.defenderStrategies[i].probabilities.Add(c1Strategy.probabilities[j] / 2);
                    result.defenderStrategies[i].partialPayoffs.Add(c1Strategy.partialPayoffs[j]);
                }
                DefenderStrategy c2Strategy = c2.defenderStrategies[i].Copy();
                for (int j = 0; j < c2Strategy.elements.Count; j++)
                {
                    result.defenderStrategies[i].elements.Add(c2Strategy.elements[j]);
                    result.defenderStrategies[i].probabilities.Add(c2Strategy.probabilities[j] / 2);
                    result.defenderStrategies[i].partialPayoffs.Add(c2Strategy.partialPayoffs[j]);
                }

                //usuwamy strategie z strategii mieszanych z prawdopodobieństem odwrotnym do prawdopodobieństwa ich wyboru
                List<int> strategiesToRemove = new List<int>(); //lista indeksów strategii z strategii mieszanych, które usuwamy

                strategiesToRemove = new List<int>();
                for (int j = 0; j < result.defenderStrategies[i].probabilities.Count; j++)
                    if (randomNumbers[j] > 0.5)
                        strategiesToRemove.Add(j);

                result.defenderStrategies[i] = result.defenderStrategies[i].Copy(strategiesToRemove);
            }

            return result;
        }



	}
}
