﻿using System.Collections.Generic;
using System.Linq;

namespace GeneticMultistepSG
{
    /// <summary>
    /// Klasa reprezentująca pojedyńcze pokolenie populacji.
    /// Zawiera osobniki oraz tworzy kolejne pokolenia za pomocą operatorów
    /// mutacji, krzyżowania i selekcji.
    public class Population
    {
        /// <summary>
        /// lista chromosomów - osobników populacji
        /// </summary>
        public List<Chromosome> chromosomes;

        /// <summary>
        /// wielkość populacji
        /// </summary>
        public int populationSize = 200;

        /// <summary>
        /// liczba najlepszych chromosomów branych do następnego pokolenia bez mutacji i krzyżowań
        /// </summary>
        public int elite = 2;

        /// <summary>
        /// współczynnik mutacji, czyli z jakim prawdopodobieństwem dany chromosom będzie zmutowany
        /// </summary>
        public double mutationRate = 0.5;

        /// <summary>
        /// liczba powtórzeń mutacji dla jednego chromosomu w ramach jednej iteracji
        /// </summary>
        public double mutationRepeats = 20;

        /// <summary>
        /// współczynnik krzyżowania, czyli z jakim prawdopodobieństwem dany chromosom będzie krzyżowany
        /// </summary>
        public double crossoverRate = 0.7;

        /// <summary>
        /// współczynnik selekcji, czyli z jakim prawdopodobieństwem jeśli chromosom wygra w turnieju to przechodzi do następnego pokolenia
        /// </summary>
        public double selectionPressure = 0.8;

        /// <summary>
        /// informacja czy ma być przeprowadzana lokalna optymalizacja
        /// </summary>
        public bool isLocalOptimization = false;

        /// <summary>
        /// wersja krzyżowania
        /// </summary>
        public int crossoverVersion = 0;

        /// <summary>
        /// wersja reprezentacji strategii w chromosomie: 0 - listy, 1 - drzewo
        /// </summary>
        public int chromosomesVersion = 0;

        /// <summary>
        /// wersje mutacji do zastosowania
        /// </summary>
        public List<int> mutationVersions = new List<int>() {0};

        // całkowita liczba mutacji
        public int mutationCount = 0;

        // całkowita liczba krzyzowan
        public int crossoverCount = 0;

        // liczba mutacji ktore poprawily wynik
        public int mutationIncreasePayoff = 0;

        // liczba krzyzowan ktore poprawily wynik
        public int crossoverIncreasePayoff = 0;


        public Population()
        {
            chromosomes = new List<Chromosome>();
        }


        public void InitPopulation()
        {
            List<Chromosome> initialChromosomes = new List<Chromosome>();
            int chromosomesCount = populationSize;
            if (Program.config.goodInitialStrategies)
                chromosomesCount = 10 * populationSize;

            for (int i = 0; i < chromosomesCount; i++)
            {
                Chromosome newChromosome;
                if (chromosomesVersion == 0)
                    newChromosome = new ChromosomeList();
                else
                    newChromosome = new ChromosomeTree();
                newChromosome.Init();
                newChromosome.Evaluate();
                initialChromosomes.Add(newChromosome);
            }

            initialChromosomes.Sort((c1, c2) => (c2.fittingFunction != c1.fittingFunction ? c2.fittingFunction.CompareTo(c1.fittingFunction) : c2.fittingFunctionSecondStage.CompareTo(c1.fittingFunctionSecondStage)));

            chromosomes = new List<Chromosome>();
            for (int i = 0; i < populationSize; i++)
                chromosomes.Add(initialChromosomes[i]);
        }


        public void MakeNewPopulation()
        {
            List<Chromosome> newChromosomes = new List<Chromosome>();

            //---ELITA---
            for (int i = 0; i < elite; i++)
            {
                newChromosomes.Add(chromosomes[i].MakeCopy());
                newChromosomes.Last().Evaluate();
            }

            if (Program.config.newIndividualsInGeneration) //w kazdej iteracji dodajemy 10% nowych osobnikow
            {
                for (int i = 0; i < populationSize * 0.1; i++)
                {
                    Chromosome c = new ChromosomeList();
                    c.Init();
                    chromosomes.Add(c);
                }
            }

            //---KRZYŻOWANIE---
            List<Chromosome> listToCrossover = new List<Chromosome>(); //lista, na którą trafią chromosomy do skrzyżowania
            List<Chromosome> listAfterCrossover = new List<Chromosome>(); //lista wszystkich chromosomów po operacji krzyżowania

            for (int i = 0; i < chromosomes.Count; i++)
            {
                if (Program.rand.NextDouble() < crossoverRate)
                    listToCrossover.Add(chromosomes[i]);
                listAfterCrossover.Add(chromosomes[i]);
            }

            listToCrossover = PermutateElements(listToCrossover); //permutujemy w celu lepszego wybierania par do krzyżowania

            for (int i = 0; i < listToCrossover.Count - 1; i += 2)
                listAfterCrossover.Add(listToCrossover[i].Crossover(listToCrossover[i], listToCrossover[i + 1], crossoverVersion));

            chromosomes = listAfterCrossover;


            //---MUTACJE---
            for (int i = 0; i < chromosomes.Count; i++)
            {
                if (Program.rand.NextDouble() < mutationRate)
                    //for (int j = 0; j < mutationRepeats; j++)
                        chromosomes[i].Mutate();
            }

            //---LOKALNA OPTYMALIZACJA---
            if (isLocalOptimization)
                foreach (Chromosome c in chromosomes)
                    c.LocalOptimization();

            //---EWALUACJA---
            foreach (Chromosome c in chromosomes)
                c.Evaluate();


            //---SELEKCJA---
            //turniejowa binarna ze zwracaniem
            while (newChromosomes.Count < populationSize) //nowe pokolenie ma tyle samo osobników
            {
                int c1 = Program.rand.Next(chromosomes.Count);
                int c2 = Program.rand.Next(chromosomes.Count);

                double k = Program.rand.NextDouble();
                if ((chromosomes[c1].fittingFunction > chromosomes[c2].fittingFunction
                    || (chromosomes[c1].fittingFunction == chromosomes[c2].fittingFunction && chromosomes[c1].fittingFunctionSecondStage > chromosomes[c2].fittingFunctionSecondStage))
                    && k < selectionPressure)
                    newChromosomes.Add(chromosomes[c1].MakeCopy());
                else
                    newChromosomes.Add(chromosomes[c2].MakeCopy());
            }

            chromosomes = newChromosomes;

            //sortowanie według funkcji przystosowania
            chromosomes.Sort((c1, c2) => (c2.fittingFunction != c1.fittingFunction ? c2.fittingFunction.CompareTo(c1.fittingFunction) : c2.fittingFunctionSecondStage.CompareTo(c1.fittingFunctionSecondStage)));
        }


        /// <summary>
        /// Permutuje losowo listę chromosomów
        /// </summary>
        public List<Chromosome> PermutateElements(List<Chromosome> list)
        {
            List<Chromosome> randomizedList = new List<Chromosome>();
            while (list.Count > 0)
            {
                int index = Program.rand.Next(0, list.Count);
                randomizedList.Add(list[index]);
                list.RemoveAt(index);
            }

            return randomizedList;
        }

    }
}
