﻿using GeneticMultistepSG.Struct;
using System;
using System.Collections.Generic;
using System.Linq;

namespace GeneticMultistepSG
{
    /// <summary> Strategie w chromosomie reprezentowane są jako drzewo strategii mieszanych obrońcy
    /// </summary>
    public class ChromosomeTree : Chromosome
    {
        /// <summary> Korzenie drzewa strategii w chromosomie (po jednym dla każdego z obrońców)
        /// </summary>
        public NodeStrategy[] strategyTrees;

        /// <summary> element [i][j] oznacza prawdopodobienstwo zlapania atakującego w interwale i w wierzcholku j
        /// </summary>
        double[][] probabilitiesAttackerCaught;


        /// <summary>
        /// Towrzy głęboką kopię chromosomu
        /// </summary>
        public override Chromosome MakeCopy()
        {
            ChromosomeTree result = new ChromosomeTree();
            result.fittingFunction = fittingFunction;
            result.fittingFunctionSecondStage = fittingFunctionSecondStage;
            result.attackerResult = attackerResult;
            if (attackStrategy == null)
                result.attackStrategy = null;
            else
                result.attackStrategy = attackStrategy.Select(x => x).ToList();

            result.strategyTrees = new NodeStrategy[strategyTrees.Length];
            for (int i = 0; i < strategyTrees.Length; i++)
                result.strategyTrees[i] = strategyTrees[i].Copy(null);

            return result;
        }


        /// <summary>
        /// Inicjalizuje chromosom (tworzy losową strategię z prawdopodobieństwami 1.0)
        /// </summary>
        public override void Init()
        {
            strategyTrees = new NodeStrategy[Program.gameDefinition.defenderUnitCount];

            for (int i = 0; i < Program.gameDefinition.defenderUnitCount; i++)
            {
                //korzeń
                strategyTrees[i] = new NodeStrategy();
                strategyTrees[i].interval = 0;
                strategyTrees[i].position = 0; //wierzchołek początkowy obrońców

                NodeStrategy previousNode = strategyTrees[i];
                NodeStrategy currentNode;

                for (int interval = 1; interval <= Program.gameDefinition.rounds; interval++) //tworzenie ścieżki w drzewie w dół
                {
                    currentNode = new NodeStrategy();
                    currentNode.parent = previousNode;
                    previousNode.children = new List<NodeStrategy>() { currentNode };
                    previousNode.probabilities = new List<double>() { 1.0 };
                    currentNode.interval = interval;
                    currentNode.position = MoveDefenderRandomly(previousNode.position);

                    previousNode = currentNode;
                }
            }
        }


        public void CalculateProbabilitiesCaughtArray()
        {
            probabilitiesAttackerCaught = new double[Program.gameDefinition.rounds + 1][];
            for (int i = 0; i < Program.gameDefinition.rounds + 1; i++)
                probabilitiesAttackerCaught[i] = new double[Program.gameDefinition.graphConfig.vertexCount];

            CalculateProbabilityCaught2(strategyTrees[0], 1.0);
        }

        /// <summary>
        /// Liczy prawdopodobieństwo 
        /// </summary>
        private void CalculateProbabilityCaught2(NodeStrategy node, double currentPathProbability)
        {
            for (int i = 0; i < node.children.Count; i++)
            {
                NodeStrategy child = node.children[i];
                probabilitiesAttackerCaught[child.interval][child.position] += currentPathProbability * node.probabilities[i];
                CalculateProbabilityCaught2(node.children[i], currentPathProbability * node.probabilities[i]);
            }
        }


        /// <summary>
        /// Liczy prawdopodobieństwo 
        /// </summary>
        private double CalculateProbabilityCaught(NodeStrategy node, int interval, double currentPathProbability, double probabilityAttackerCaught, List<int> attackerStrategy)
        {
            double suma = 0.0;
            if (node.interval > interval)
                return 0.0;

            for (int i = 0; i < node.children.Count; i++)
            {
                NodeStrategy child = node.children[i];
                if (child.interval == interval)
                {
                    if (attackerStrategy[child.interval] == child.position)
                        suma += currentPathProbability * node.probabilities[i];
                }
                else
                    suma += CalculateProbabilityCaught(node.children[i], interval, currentPathProbability * node.probabilities[i], probabilityAttackerCaught, attackerStrategy);
            }

            return suma;
        }

        public override void CalculatePartialPayoffs(List<int> attackerStrategy)
        {
            throw new NotImplementedException();
        }


        /// <summary>
        /// Wyliczenie wypłaty atakującego (attackerResult) i obrońcy (defenderResult) dla strategii atakującego attackerStrategy,
        /// z uwględnieniem ograniczonej racjonalności brStrategy. Strategia obrońcy z chromosomu.
        /// </summary>
        public override void CalculatePayoff(List<int> attackerStrategy, List<BoundedRationalityStrategy> brStrategy, out double attackerResult, out double defenderResult)
        {
            strategyTrees[0].Normalize();

            attackerResult = 0.0; defenderResult = 0.0;
            double currentProbablility = 1.0;

            NodeStrategy[] strategyTreesToEvaluate = new NodeStrategy[strategyTrees.Length];

            strategyTreesToEvaluate = strategyTrees;

            //TODO: uwzględnić boundedRationality


            for (int interval = 0; interval < attackerStrategy.Count; interval++)
            {
                int v = attackerStrategy[interval];

                //double probabilityAttackerCaught = CalculateProbabilityCaught(strategyTrees[0], interval, 1.0, 0.0, attackerStrategy); //TODO: uwzględnić więcej niż 1 obrońcę
                double probabilityAttackerCaught = probabilitiesAttackerCaught[interval][v];

                if (probabilityAttackerCaught > 1 || probabilityAttackerCaught < 0)
                    probabilityAttackerCaught = probabilityAttackerCaught;

                double defenderReward = Program.gameDefinition.vertexDefenderRewards[v];
                double attackerPenality = Program.gameDefinition.vertexAttackerPenalties[v];

                if (brStrategy.Contains(BoundedRationalityStrategy.ProspectTheoryPayoff))
                {
                    defenderReward = BoundedRationality.ProspectTheoryPayoff(defenderReward);
                    attackerPenality = BoundedRationality.ProspectTheoryPayoff(attackerPenality);
                }

                defenderResult += currentProbablility * probabilityAttackerCaught * defenderReward;
                attackerResult += currentProbablility * probabilityAttackerCaught * attackerPenality;

                if (Program.gameDefinition.targets.Contains(v)) //znajdujemy się w celu
                {
                    double defenderPanality = Program.gameDefinition.targetDefenderPenalties[Program.gameDefinition.targets.IndexOf(v)];
                    double attackerReward = Program.gameDefinition.targetAttackerRewards[Program.gameDefinition.targets.IndexOf(v)];

                    if (brStrategy.Contains(BoundedRationalityStrategy.ProspectTheoryPayoff))
                    {
                        defenderPanality = BoundedRationality.ProspectTheoryPayoff(defenderPanality);
                        attackerReward = BoundedRationality.ProspectTheoryPayoff(attackerReward);
                    }

                    defenderResult += currentProbablility * (1 - probabilityAttackerCaught) * defenderPanality;
                    attackerResult += currentProbablility * (1 - probabilityAttackerCaught) * attackerReward;
                }

                currentProbablility = currentProbablility * (1 - probabilityAttackerCaught); //prawdopodobieństwo, że atakujący nie został złapany i idzie dalej

            }
        }


        public override void Mutate()
        {
            Mutate0();
        }


        /// <summary>
        /// Na losowym poziomie drzewa zmieniamy losowo jedno prawdopodobieństwo, pozostałe normalizujemy
        /// </summary>
        public void Mutate0()
        {
            int defenderToMute = Program.rand.Next(strategyTrees.Length);
            int intervalToMute = Program.rand.Next(Program.gameDefinition.rounds);

            NodeStrategy nodeToMute = strategyTrees[defenderToMute];
            for (int i = 1; i < intervalToMute - 1; i++)
                nodeToMute = nodeToMute.children[Program.rand.Next(nodeToMute.children.Count)];

            nodeToMute.probabilities[Program.rand.Next(nodeToMute.probabilities.Count)] = Program.rand.NextDouble();

            //normalizacja prawdopodobieństw
            double probabilitiesSum = nodeToMute.probabilities.Sum();
            nodeToMute.probabilities = nodeToMute.probabilities.Select(x => x / probabilitiesSum).ToList();
        }

        /// <summary>
        /// Na losowym poziomie drzewa dodajemy losową ścieżkę w dół (z losowymi prawdopodobieństwami)
        /// </summary>
        public void Mutate1()
        {
            int defenderToMute = Program.rand.Next(strategyTrees.Length);
            int intervalToMute = Program.rand.Next(Program.gameDefinition.rounds);

            NodeStrategy nodeToMute = strategyTrees[defenderToMute];
            for (int i = 1; i < intervalToMute - 1; i++)
                nodeToMute = nodeToMute.children[Program.rand.Next(nodeToMute.children.Count)];

            //korzeń
            NodeStrategy newNode = new NodeStrategy();
            newNode.interval = intervalToMute;
            newNode.position = nodeToMute.position;

            NodeStrategy previousNode = newNode;
            NodeStrategy currentNode;

            for (int interval = intervalToMute + 1; interval <= Program.gameDefinition.rounds; interval++) //tworzenie ścieżki w drzewie w dół
            {
                currentNode = new NodeStrategy();
                currentNode.parent = previousNode;
                previousNode.children = new List<NodeStrategy>() { currentNode };
                previousNode.probabilities = new List<double>() { Program.rand.NextDouble() };
                currentNode.interval = interval;
                currentNode.position = MoveDefenderRandomly(previousNode.position);

                previousNode = currentNode;
            }

            nodeToMute = Merge(nodeToMute, newNode, nodeToMute.parent);
        }

        /// <summary>
        /// Usuwamy losową ścieżkę w dół (czyli jedną strategię prostą)
        /// </summary>
        public void Mutate2()
        {
            int defenderToMute = Program.rand.Next(strategyTrees.Length);
            int intervalToMute = Program.rand.Next(Program.gameDefinition.rounds);

            NodeStrategy nodeToMute = strategyTrees[defenderToMute];
            for (int i = 1; i < intervalToMute - 1; i++)
                nodeToMute = nodeToMute.children[Program.rand.Next(nodeToMute.children.Count)];

            if (nodeToMute.children.Count > 1)
            {
                int indexToDelete = Program.rand.Next(nodeToMute.children.Count);
                nodeToMute.children.RemoveAt(indexToDelete);
                nodeToMute.probabilities.RemoveAt(indexToDelete);
            }

            //normalizacja prawdopodobieństw
            double probabilitiesSum = nodeToMute.probabilities.Sum();
            nodeToMute.probabilities = nodeToMute.probabilities.Select(x => x / probabilitiesSum).ToList();
        }


        public override Chromosome Crossover(Chromosome c1, Chromosome c2, int version)
        {
            if (version == 0)
                return Crossover0(c1 as ChromosomeTree, c2 as ChromosomeTree);
            else if (version == 1)
                return Crossover1(c1 as ChromosomeTree, c2 as ChromosomeTree);
            else
                throw new NotImplementedException("Brak krzyzowania w wersji " + version);
        }


        /// <summary>
        /// Łączymy drzewa z c1 i c2 i poprawiamy prawdopodobieństwa
        /// </summary>
        public ChromosomeTree Crossover0(ChromosomeTree c1, ChromosomeTree c2)
        {
            ChromosomeTree newChromosome = new ChromosomeTree();
            newChromosome.strategyTrees = new NodeStrategy[Program.gameDefinition.defenderUnitCount];
            for (int i = 0; i < newChromosome.strategyTrees.Length; i++)
                newChromosome.strategyTrees[i] = Merge(c1.strategyTrees[i], c2.strategyTrees[i], null);

            return newChromosome;
        }

        /// <summary>
        /// Łączymy drzewa z c1 i c2 i poprawiamy prawdopodobieństwa, a następnie redukujemy rozmiar drzewa
        /// </summary>
        public ChromosomeTree Crossover1(ChromosomeTree c1, ChromosomeTree c2)
        {
            ChromosomeTree newChromosome = new ChromosomeTree();
            newChromosome.strategyTrees = new NodeStrategy[Program.gameDefinition.defenderUnitCount];
            for (int i = 0; i < newChromosome.strategyTrees.Length; i++)
            {
                newChromosome.strategyTrees[i] = Merge(c1.strategyTrees[i], c2.strategyTrees[i], null);
                ReduceTreeSize(newChromosome.strategyTrees[i]);
            }

            return newChromosome;
        }

        public NodeStrategy Merge(NodeStrategy s1, NodeStrategy s2, NodeStrategy parent)
        {
            if (s1.position != s2.position)
                throw new ArgumentException("Próba połączenia węzłów z innym polem");

            NodeStrategy mergedNode = s1;
            for (int i = 0; i < s2.children.Count; i++)
            {
                NodeStrategy nodeToAdd = s2.children[i];
                if (!mergedNode.children.Any(x => x.position == nodeToAdd.position)) //nie ma następnika o tej samej pozycji
                {
                    mergedNode.parent = parent;
                    mergedNode.children.Add(nodeToAdd);
                    mergedNode.probabilities.Add(s2.probabilities[i]);
                }
                else //istnieje następnik o tej samej pozycji
                {
                    int indx = mergedNode.children.FindIndex(x => x.position == nodeToAdd.position); //indeks węzła z tym samym dzieckiem

                    mergedNode.probabilities[indx] += s1.probabilities[i];
                    mergedNode.children[indx] = Merge(mergedNode.children[indx], nodeToAdd, mergedNode);
                }

            }

            //normalizacja prawdopodobieństw
            double probabilitiesSum = mergedNode.probabilities.Sum();
            mergedNode.probabilities = mergedNode.probabilities.Select(x => x / probabilitiesSum).ToList();

            return mergedNode;
        }


        /// <summary>
        /// Zmniejszenie rozmiaru drzewa strategii.
        /// Gałąź jest usuwana z prawdopodobieństwem odwrotnym do pierwiastka prawdopodobieństwa wyboru danej strategii
        /// </summary>
        public void ReduceTreeSize(NodeStrategy s)
        {
            int luckyChild = Program.rand.Next(s.children.Count); //co najmniej jeden następnik zawsze zostaje
            List<int> childrenToDelete = new List<int>(); //lista indeksów węzłów-dzieci, które będą usuwane
            for (int i = 0; i < s.children.Count; i++)
            {
                if (i != luckyChild && Program.rand.NextDouble() * Program.rand.NextDouble() > s.probabilities[i])
                    childrenToDelete.Add(i);
            }

            for (int i = childrenToDelete.Count - 1; i >= 0; i--)
            {
                s.children.RemoveAt(childrenToDelete[i]);
                s.probabilities.RemoveAt(childrenToDelete[i]);
            }

            //normalizacja prawdopodobieństw
            double probabilitiesSum = s.probabilities.Sum();
            s.probabilities = s.probabilities.Select(x => x / probabilitiesSum).ToList();

            foreach (NodeStrategy child in s.children)
                ReduceTreeSize(child);
        }


        public override void LocalOptimization()
        {

        }

    }
}