﻿using GeneticMultistepSG.Struct;
using System;
using System.Collections.Generic;
using System.Linq;

namespace GeneticMultistepSG
{
    /// <summary> Klasa abstrakcyjna reprezentująca chromosom
    /// </summary>
    public abstract class Chromosome
    {
        /// <summary> funkcja przystosowania, oznacza jak dobry jest dany chromosom (wartość oczekiwana obrońcy)
        /// </summary>
        public double fittingFunction;

        /// <summary> funkcja przystosowania, która jest brana pod uwagę przy porównywaniu chromosomów w przypadku 
        /// równej wartości fittingFunction
        /// </summary>
        public double fittingFunctionSecondStage;

        /// <summary> oczekiwany wynik atakującego przy ataku w optymalnym punkcie
        /// </summary>
        public double attackerResult;

        /// <summary> strategia atakującego
        /// </summary>
        public List<int> attackStrategy;

        // liczba zamian optymalnego atakujacego
        public bool isDifferentAttacker = false;

        /// <summary> lista wersji ograniczonej racjonalności, które stosowane są w grze
        /// </summary>
        public List<BoundedRationalityStrategy> boundedRationality
            = new List<BoundedRationalityStrategy>() { /*BoundedRationalityStrategy.AnchoringTheoryProbabilityApproximation*/ };


        public abstract void Init();

        public abstract Chromosome MakeCopy();

        public abstract Chromosome Crossover(Chromosome c1, Chromosome c2, int version);

        public abstract void Mutate();

        public abstract void CalculatePayoff(List<int> attackerStrategy, List<BoundedRationalityStrategy> brStrategy, out double attackerResult, out double defenderResult);

        public abstract void CalculatePartialPayoffs(List<int> attackerStrategy);

        public abstract void LocalOptimization();

        /// <summary>
        /// Oblicza funkcję przystosowania chromosomu - wynik obrońcy przy zastosowaniu strategii mieszanej zakodowanej w chromosomie
        /// </summary>
        public void Evaluate()
        {
            List<int> previousAttackStrategy = new List<int>();
            if (attackStrategy != null)
                previousAttackStrategy = attackStrategy.Select(x => x).ToList();

            if (this is ChromosomeTree)
                (this as ChromosomeTree).CalculateProbabilitiesCaughtArray();

            List<int> bestAttackerStrategy = new List<int>();
            double bestResultAttacker = -int.MaxValue;
            double defenderResultForBestAttackerMove = -int.MaxValue;
            fittingFunctionSecondStage = 0;

            //dla atakującego sprawdzamy rezultat ataku każdego z celów w każdym z interwałów czasowych
            foreach (List<int> attackerStrategy in Program.gameDefinition.attackerStrategies)
            {
                double attackerResult = 0.0, defenderResult = 0.0;

                CalculatePayoff(attackerStrategy, boundedRationality, out attackerResult, out defenderResult);

                if (attackerResult == 0.0)
                {
                    attackerResult = 0;
                    defenderResult = 0;
                }

                if (attackerResult > bestResultAttacker || (Math.Abs(attackerResult - bestResultAttacker) < Program.EPS && defenderResult > defenderResultForBestAttackerMove))
                {
                    bestResultAttacker = attackerResult;
                    defenderResultForBestAttackerMove = defenderResult;
                    bestAttackerStrategy = attackerStrategy;
                }
            }

            CalculatePayoff(bestAttackerStrategy, new List<BoundedRationalityStrategy>(), out this.attackerResult, out fittingFunction);
            CalculatePartialPayoffs(bestAttackerStrategy);

            //fittingFunction = defenderResultForBestAttackerMove;
            //this.attackerResult = bestResultAttacker;
            attackStrategy = bestAttackerStrategy;

            if (string.Join(" ", previousAttackStrategy) == string.Join(" ", attackStrategy))
                isDifferentAttacker = false;
            else
                isDifferentAttacker = true;
        }


        public int MoveDefenderRandomly(int v)
        {
            int neighboursCount = Program.gameDefinition.graphConfig.adjacencyList[v].Count;

            List<int> neighbourTargets = Program.gameDefinition.graphConfig.adjacencyList[v].Intersect(Program.gameDefinition.targets).ToList();

            /*if (neighbourTargets.Count > 0 && Program.rand.Next(3) == 0)
                return neighbourTargets[Program.rand.Next(neighbourTargets.Count)];
            else*/
            return Program.gameDefinition.graphConfig.adjacencyList[v][Program.rand.Next(neighboursCount)];
        }
    }
}
